<?php


function processRemoteImageToFeatured($imageUrl, $post_id, $websiteName = 'LITONEWS') {
    require_once(ABSPATH . 'wp-admin/includes/image.php');
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    require_once(ABSPATH . 'wp-admin/includes/media.php');

    // Step 1: Download image data
    $imageData = file_get_contents($imageUrl);
    if ($imageData === false) {
        die("Failed to download image from URL.");
    }

    // Step 2: Create GD image
    $image = imagecreatefromstring($imageData);
    if (!$image) {
        die("Failed to create image from data.");
    }

    $width = imagesx($image);
    $height = imagesy($image);

    // Step 3: Crop 10px from all sides
    $cropSize = 50;
    $cropped = imagecrop($image, [
        'x' => $cropSize,
        'y' => $cropSize,
        'width' => $width - 2 * $cropSize,
        'height' => $height - 2 * $cropSize,
    ]);
    imagedestroy($image);
    if (!$cropped) {
        die("Cropping failed.");
    }

    // Step 4: Mirror image
    $flipped = imagecreatetruecolor(imagesx($cropped), imagesy($cropped));
    for ($x = 0; $x < imagesx($cropped); $x++) {
        imagecopy($flipped, $cropped, imagesx($cropped) - $x - 1, 0, $x, 0, 1, imagesy($cropped));
    }
    imagedestroy($cropped);

    // Step 5: Add watermark
    $textColor = imagecolorallocate($flipped, 255, 255, 255);
    $fontSize = 70;
    $margin = 10;
    $textWidth = imagefontwidth($fontSize) * strlen($websiteName);
    $textHeight = imagefontheight($fontSize);
    $x = imagesx($flipped) - $textWidth - $margin;
    $y = imagesy($flipped) - $textHeight - $margin;
    imagestring($flipped, $fontSize, $x, $y, $websiteName, $textColor);

    // Step 6: Save the image to a temp file
    $upload_dir = wp_upload_dir();
    $filename = 'featured_' . time() . '.jpg';
    $tmp_path = $upload_dir['path'] . '/' . $filename;
    imagejpeg($flipped, $tmp_path, 90);
    imagedestroy($flipped);

    // Step 7: Upload to WordPress Media Library
    $file_array = [
        'name'     => basename($filename),
        'tmp_name' => $tmp_path,
    ];

    $attachment_id = media_handle_sideload($file_array, $post_id);
    if (is_wp_error($attachment_id)) {
        @unlink($tmp_path);
        die('Media upload failed: ' . $attachment_id->get_error_message());
    }

    // Step 8: Set as featured image
    set_post_thumbnail($post_id, $attachment_id);
}

// Fetch trending topics from Google Trends
function getTrendingTopics($feedUrl) {
    $rss = simplexml_load_file($feedUrl, 'SimpleXMLElement', LIBXML_NOCDATA);
    $namespaces = $rss->getNamespaces(true);
    $topics = [];

    foreach ($rss->channel->item as $entry) {
        $ht = $entry->children($namespaces['ht']);
        $keyword = (string)$entry->title;
        $titles = [];
        $urls = []; 
        foreach ($ht->news_item as $news) {
           
            $titles[] = (string)$news->news_item_title;
            $urls[] = (string)$news->news_item_url;
        }
        
           
        if (empty($titles)) {
           array_push($titles, $keyword);
        }

        $topics[] = [
            'keyword' => $keyword,
            'titles' => $titles,
            'urls'  => $urls
        ];
    }

    return $topics;
}

function buildCategoryPrompt($keyword, $titles = []) {
    $context = implode("; ", $titles);
    $categories = [
    "Breaking News", "Politics", "Entertainment", "Music", "Movies & TV Shows",
    "Celebrities", "Technology", "Science", "Health & Wellness", "Sports",
    "Fashion & Style", "Beauty & Skincare", "Culture & Lifestyle", "Social Media Trends",
    "Viral Videos", "Business & Economy", "Finance & Investing", "Education", "Gaming",
    "Travel & Tourism", "Food & Recipes", "Weather & Natural Disasters", "Crime & Law",
    "Memes & Internet Culture", "Events & Festivals", "Opinion & Editorials", "World News",
    "Startups & Innovation", "AI & Future Tech", "Mobile & Gadgets", "Apps & Software",
    "Product Launches", "Leaked News & Rumors", "TV Dramas & Series", "Movie Trailers & Teasers",
    "Reality TV", "Award Shows", "Influencers & Creators", "YouTube Trends", "TikTok Trends",
    "Instagram Trends", "X (Twitter) Trends", "Startup Funding & Acquisitions",
    "Cryptocurrency & Blockchain", "Artificial Intelligence", "Job Market & Careers",
    "Online Earning", "Scandals & Controversies", "Relationship & Dating News",
    "Celebrity Deaths & Tributes", "Elections & Campaigns", "Protests & Social Movements",
    "Global Conflicts & War", "Environment & Climate", "Inventions & Discoveries",
    "Public Reactions & Comments", "Internet Challenges", "Fashion Week Highlights",
    "Fitness Trends", "Home & Interior Trends", "Books & Literature News", "Horoscope & Zodiac",
    "Religion & Spirituality", "Transportation & Auto News"
];
$allcategories = implode("; ",$categories);
    return <<<PROMPT
You are a WordPress SEO assistant.

Based on the trending keyword: "$keyword"  
And the following related news headlines: $context
and here is the list of categories: $allcategories

Suggest the most suitable WordPress blog category name from this list.  

Only return the category name. No extra words or explanation.
PROMPT;
}


// Build prompt for SEO title/meta
function buildSeoTitleMetaPrompt($keyword, $titles = []) {
    $context = implode("; ", $titles);
    return <<<PROMPT
You are an SEO content strategist.

Based on the trending keyword: "$keyword"  
and the following related headlines: $context

Generate ONLY the following output in this **exact format**, so it can be parsed easily:

{{SEO_TITLE}}: Your SEO-optimized title here (max 60 characters, must include the keyword)  
{{META_DESCRIPTION}}: Your SEO-optimized meta description here (max 160 characters, clear and engaging summary)

Only include the above two lines in the response.
PROMPT;
}
// full article
function buildFullArticlePrompt($keyword, $titles = [], $seoTitle = '') {
    $context = implode("; ", $titles);
    return <<<PROMPT
Write a blog article with the title: "$seoTitle".

Context: $context

Instructions:
- Use simple and native-friendly language
- Write in clean HTML format (no markdown, no asterisks, no backticks)
- Do NOT include any markdown code blocks or ```html. Only return raw HTML content. No code fences.
- Do not add any heading before the first paragraph.
- Structure with proper HTML tags like <h2>, <h3>, <p>, <ul>, <li>
- Start with an engaging introduction
- Use subheadings, bullet points, and short paragraphs
- End with a strong conclusion
- Use "$seoTitle" as the article title (do NOT use <h1>)
- Use clear, simple, and easy-to-understand English
- Avoid complicated vocabulary and technical words
- Break down ideas clearly so anyone can understand
- Use short paragraphs and friendly tone
- Organize the content using <h2> and <h3> only
- Include bullet points or numbered lists if helpful
- Start with a simple introduction that grabs attention
- End with a clear and helpful conclusion
- Answer common search questions in FAQ style if needed
- Make it SEO-optimized but human-friendly
- Do NOT include <h1> or title/meta HTML — just the article body in HTML
- Target length: 1500 to 2000 words
- Avoid AI-related wording like “as an AI” or “here’s your article”
- Do NOT use any placeholder text like [Player Name], [Country], or [Event Date] etc
- Replace all placeholders with real names and actual known details
- If information is not available, rewrite the sentence without mentioning it
- NEVER leave brackets or incomplete details in the text

PROMPT;
}



// Ask Gemini API (instead of OpenAI)
function askGemini($prompt, $apiKey) {
    $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=$apiKey";

    $postData = json_encode([
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ]
    ]);

    $headers = [
        'Content-Type: application/json'
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    $response = curl_exec($ch);
    if (curl_errno($ch)) {
        throw new Exception('Curl error: ' . curl_error($ch));
    }
    curl_close($ch);

    $responseData = json_decode($response, true);

    return $responseData['candidates'][0]['content']['parts'][0]['text'] ?? '';
}

// Extract SEO title & meta
function extractSeoData($response) {
    preg_match('/{{SEO_TITLE}}:\s*(.+)/', $response, $titleMatch);
    preg_match('/{{META_DESCRIPTION}}:\s*(.+)/', $response, $descMatch);

    return [
        'seo_title' => $titleMatch[1] ?? '',
        'meta_description' => $descMatch[1] ?? ''
    ];
}

function focusKeywordExists($keyword) {
    global $wpdb;

    $meta_key = '_yoast_wpseo_focuskw';
    $query = $wpdb->prepare(
        "SELECT post_id FROM $wpdb->postmeta WHERE meta_key = %s AND meta_value = %s LIMIT 1",
        $meta_key,
        $keyword
    );

    $result = $wpdb->get_var($query);
    return !empty($result);
}


function insert_gemini_article_to_wp($post_title, $post_content, $focus_keyword, $seo_description, $category_name = 'Blog') {
    // 1. Create or get the category ID
    $category_id = get_cat_ID($category_name);
    if ($category_id === 0) {
        $new_category = wp_create_category($category_name);
        if (!is_wp_error($new_category)) {
            $category_id = $new_category;
        }
    }

    // 2. Insert the post
    $post_data = [
        'post_title'    => $post_title,
        'post_content'  => trim(str_ireplace(array("```html","```"), '', $post_content)),
        'post_status'   => 'publish', // Change to 'draft' if needed
        'post_author'   => 1,         // Author ID
        'post_category' => [$category_id]
    ];

    $post_id = wp_insert_post($post_data);

    if (!is_wp_error($post_id)) {
        // 3. Add Yoast SEO meta fields
        update_post_meta($post_id, '_yoast_wpseo_title', $post_title);              // Yoast SEO title
        update_post_meta($post_id, '_yoast_wpseo_metadesc', $seo_description);      // Meta description
        update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword);         // Focus keyword
        update_post_meta($post_id, '_yoast_wpseo_focuskw_text_input', $focus_keyword); // UI input

        return $post_id;
    } else {
        return new WP_Error('insert_failed', 'Post could not be inserted.');
    }
}

function getOgImage($url) {
    // Initialize cURL session
    $ch = curl_init();

    // Set cURL options
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; PHP-cURL)',
        CURLOPT_TIMEOUT => 10,
    ]);

    // Execute cURL and get the HTML content
    $html = curl_exec($ch);
    curl_close($ch);

    // Check if fetch was successful
    if (!$html) {
        return false;
    }

    // Use DOMDocument to parse HTML
    libxml_use_internal_errors(true); // suppress parsing warnings
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    libxml_clear_errors();

    $xpath = new DOMXPath($doc);
    $metaTags = $xpath->query("//meta[@property='og:image']");

    if ($metaTags->length > 0) {
        return $metaTags->item(0)->getAttribute('content');
    }

    return false; // OG image not found
}



